<?php
// src/Infrastructure/Payment/MpesaGateway.php
class MpesaGateway {
    private $consumerKey;
    private $consumerSecret;
    private $businessShortCode;
    private $passkey;
    private $callbackUrl;
    private $environment;
    
    public function __construct($config) {
        $this->consumerKey = $config['consumer_key'];
        $this->consumerSecret = $config['consumer_secret'];
        $this->businessShortCode = $config['business_short_code'];
        $this->passkey = $config['passkey'];
        $this->callbackUrl = $config['callback_url'];
        $this->environment = $config['environment'] ?? 'sandbox';
    }
    
    private function getAccessToken() {
        $url = $this->getBaseUrl() . '/oauth/v1/generate?grant_type=client_credentials';
        $credentials = base64_encode($this->consumerKey . ':' . $this->consumerSecret);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception('Failed to get M-Pesa access token');
        }
        
        $data = json_decode($response, true);
        return $data['access_token'];
    }
    
    private function getBaseUrl() {
        return $this->environment === 'production' 
            ? 'https://api.safaricom.co.ke' 
            : 'https://sandbox.safaricom.co.ke';
    }
    
    private function generatePassword() {
        $timestamp = date('YmdHis');
        return base64_encode($this->businessShortCode . $this->passkey . $timestamp);
    }
    
    public function initiateSTKPush(PhoneNumber $phoneNumber, Money $amount, TransactionId $transactionId, $accountReference, $transactionDesc) {
        $accessToken = $this->getAccessToken();
        $timestamp = date('YmdHis');
        $password = $this->generatePassword();
        
        $url = $this->getBaseUrl() . '/mpesa/stkpush/v1/processrequest';
        
        $payload = [
            'BusinessShortCode' => $this->businessShortCode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'TransactionType' => 'CustomerPayBillOnline',
            'Amount' => (int) $amount->getAmount(),
            'PartyA' => $phoneNumber->getFormattedNumber(),
            'PartyB' => $this->businessShortCode,
            'PhoneNumber' => $phoneNumber->getFormattedNumber(),
            'CallBackURL' => $this->callbackUrl,
            'AccountReference' => $accountReference,
            'TransactionDesc' => $transactionDesc
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception('STK Push request failed');
        }
        
        return json_decode($response, true);
    }
    
    public function handleCallback($callbackData) {
        $stkCallback = $callbackData['Body']['stkCallback'] ?? null;
        
        if (!$stkCallback) {
            throw new Exception('Invalid callback structure');
        }
        
        $resultCode = $stkCallback['ResultCode'];
        $resultDesc = $stkCallback['ResultDesc'];
        $merchantRequestID = $stkCallback['MerchantRequestID'];
        $checkoutRequestID = $stkCallback['CheckoutRequestID'];
        
        $result = [
            'result_code' => $resultCode,
            'result_desc' => $resultDesc,
            'merchant_request_id' => $merchantRequestID,
            'checkout_request_id' => $checkoutRequestID,
            'success' => $resultCode == 0
        ];
        
        if ($resultCode == 0 && isset($stkCallback['CallbackMetadata']['Item'])) {
            $metadata = $stkCallback['CallbackMetadata']['Item'];
            
            foreach ($metadata as $item) {
                switch ($item['Name']) {
                    case 'Amount':
                        $result['amount'] = $item['Value'];
                        break;
                    case 'MpesaReceiptNumber':
                        $result['mpesa_receipt_number'] = $item['Value'];
                        break;
                    case 'TransactionDate':
                        $result['transaction_date'] = $item['Value'];
                        break;
                    case 'PhoneNumber':
                        $result['phone_number'] = $item['Value'];
                        break;
                }
            }
        }
        
        return $result;
    }
}