<?php
// src/Infrastructure/Repositories/MySQLOrderRepository.php
class MySQLOrderRepository implements OrderRepositoryInterface {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    public function save(Order $order) {
        try {
            $this->db->beginTransaction();
            
            $stmt = $this->db->prepare("
                INSERT INTO orders (order_number, customer_id, total_items, total_amount, status, created_at) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $order->getOrderNumber(),
                $order->getCustomerId(),
                $order->getTotalItems(),
                $order->getTotalAmount(),
                $order->getStatus(),
                $order->getCreatedAt()->format('Y-m-d H:i:s')
            ]);
            
            $orderId = $this->db->lastInsertId();
            $order->setId($orderId);
            
            // Save order items
            foreach ($order->getItems() as $item) {
                $itemStmt = $this->db->prepare("
                    INSERT INTO order_items (order_id, scheme_id, created_at) 
                    VALUES (?, ?, ?)
                ");
                $itemStmt->execute([
                    $orderId,
                    $item->getSchemeId(),
                    $item->getCreatedAt()->format('Y-m-d H:i:s')
                ]);
            }
            
            $this->db->commit();
            return $order;
            
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function findById($id) {
        $stmt = $this->db->prepare("SELECT * FROM orders WHERE id = ?");
        $stmt->execute([$id]);
        $data = $stmt->fetch();
        
        if (!$data) return null;
        
        return $this->hydrate($data);
    }
    
    public function findByOrderNumber($orderNumber) {
        $stmt = $this->db->prepare("SELECT * FROM orders WHERE order_number = ?");
        $stmt->execute([$orderNumber]);
        $data = $stmt->fetch();
        
        if (!$data) return null;
        
        return $this->hydrate($data);
    }
    
    public function findByCustomerId($customerId) {
        $stmt = $this->db->prepare("
            SELECT * FROM orders 
            WHERE customer_id = ? 
            ORDER BY created_at DESC
        ");
        $stmt->execute([$customerId]);
        $results = $stmt->fetchAll();
        
        $orders = [];
        foreach ($results as $data) {
            $orders[] = $this->hydrate($data);
        }
        
        return $orders;
    }
    
    public function update(Order $order) {
        $stmt = $this->db->prepare("
            UPDATE orders 
            SET status = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$order->getStatus(), $order->getId()]);
        
        return $order;
    }
    
    private function hydrate($data) {
        $order = new Order(
            $data['order_number'],
            $data['customer_id'],
            $data['total_items'],
            $data['total_amount']
        );
        
        $order->setId($data['id']);
        $order->setStatus($data['status']);
        
        // Load order items
        $this->loadOrderItems($order);
        
        return $order;
    }
    
    private function loadOrderItems(Order $order) {
        $stmt = $this->db->prepare("
            SELECT * FROM order_items WHERE order_id = ?
        ");
        $stmt->execute([$order->getId()]);
        $items = $stmt->fetchAll();
        
        foreach ($items as $itemData) {
            $item = new OrderItem($itemData['order_id'], $itemData['scheme_id']);
            $item->setId($itemData['id']);
            $order->addItem($item);
        }
    }
}