<?php
class MySQLTransactionRepository implements TransactionRepositoryInterface {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    public function save(Transaction $transaction) {
        $stmt = $this->db->prepare("
            INSERT INTO transactions (transaction_id, order_id, amount, phone_number, status, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $transaction->getTransactionId(),
            $transaction->getOrderId(),
            $transaction->getAmount(),
            $transaction->getPhoneNumber(),
            $transaction->getStatus(),
            $transaction->getCreatedAt()->format('Y-m-d H:i:s'),
            $transaction->getUpdatedAt()->format('Y-m-d H:i:s')
        ]);
        
        $transaction->setId($this->db->lastInsertId());
        return $transaction;
    }
    
    public function findById($id) {
        $stmt = $this->db->prepare("SELECT * FROM transactions WHERE id = ?");
        $stmt->execute([$id]);
        $data = $stmt->fetch();
        
        if (!$data) return null;
        
        return $this->hydrate($data);
    }
    
    public function findByTransactionId($transactionId) {
        $stmt = $this->db->prepare("SELECT * FROM transactions WHERE transaction_id = ?");
        $stmt->execute([$transactionId]);
        $data = $stmt->fetch();
        
        if (!$data) return null;
        
        return $this->hydrate($data);
    }
    
    public function findByCheckoutRequestId($checkoutRequestId) {
        $stmt = $this->db->prepare("SELECT * FROM transactions WHERE checkout_request_id = ?");
        $stmt->execute([$checkoutRequestId]);
        $data = $stmt->fetch();
        
        if (!$data) return null;
        
        return $this->hydrate($data);
    }
    
    public function update(Transaction $transaction) {
        $stmt = $this->db->prepare("
            UPDATE transactions 
            SET status = ?, mpesa_receipt_number = ?, checkout_request_id = ?, updated_at = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $transaction->getStatus(),
            $transaction->getMpesaReceiptNumber(),
            $transaction->getCheckoutRequestId(),
            $transaction->getUpdatedAt()->format('Y-m-d H:i:s'),
            $transaction->getId()
        ]);
        
        return $transaction;
    }
    
    private function hydrate($data) {
        $transaction = new Transaction(
            $data['transaction_id'],
            $data['order_id'],
            $data['amount'],
            $data['phone_number']
        );
        
        $transaction->setId($data['id']);
        
        if ($data['mpesa_receipt_number']) {
            $transaction->setMpesaReceiptNumber($data['mpesa_receipt_number']);
        }
        
        if ($data['checkout_request_id']) {
            $transaction->setCheckoutRequestId($data['checkout_request_id']);
        }
        
        // Set status based on data
        switch ($data['status']) {
            case Transaction::STATUS_COMPLETED:
                $transaction->markAsCompleted($data['mpesa_receipt_number']);
                break;
            case Transaction::STATUS_FAILED:
                $transaction->markAsFailed();
                break;
        }
        
        return $transaction;
    }
}
?>